/*******************************************************************************
 * Copyright (c) 2000, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.swt.widgets;

import intrinsic.Function;
import intrinsic.mx.controls.*;
import intrinsic.mx.core.EdgeMetrics;
import intrinsic.mx.core.ScrollControlBase;
import intrinsic.mx.events.FlexEvent;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.events.*;

public class Text extends Scrollable {

	public static final int LIMIT = 0x7FFFFFFF;
	public static final String DELIMITER = "\n";
	
public Text (Composite parent, int style) {
	super (parent, checkStyle (style));
}

public void addModifyListener (ModifyListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Modify, typedListener);
}

public void addSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Selection,typedListener);
	addListener (SWT.DefaultSelection,typedListener);
}

public void addVerifyListener (VerifyListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	TypedListener typedListener = new TypedListener (listener);
	addListener (SWT.Verify, typedListener);
}

public void append (String string) {
	checkWidget ();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	if ((style & SWT.SINGLE) != 0) {
		((TextInput)object).text += string;
	} else {
		((TextArea)object).text += string;		
	}
}

static int checkStyle (int style) {
	if ((style & SWT.SEARCH) != 0) {
		style |= SWT.SINGLE | SWT.BORDER;
		style &= ~SWT.PASSWORD;
	}
	style &= ~SWT.SEARCH;
	if ((style & SWT.SINGLE) != 0 && (style & SWT.MULTI) != 0) {
		style &= ~SWT.MULTI;
	}
	style = checkBits (style, SWT.LEFT, SWT.CENTER, SWT.RIGHT, 0, 0, 0);
	if ((style & SWT.SINGLE) != 0) style &= ~(SWT.H_SCROLL | SWT.V_SCROLL | SWT.WRAP);
	if ((style & SWT.WRAP) != 0) {
		style |= SWT.MULTI;
		style &= ~SWT.H_SCROLL;
	}
	if ((style & SWT.MULTI) != 0) style &= ~SWT.PASSWORD;
	if ((style & (SWT.SINGLE | SWT.MULTI)) != 0) return style;
	if ((style & (SWT.H_SCROLL | SWT.V_SCROLL)) != 0) return style | SWT.MULTI;
	return style | SWT.SINGLE;
}


public void clearSelection () {
	checkWidget ();
	Point selection = getSelection();
	setSelection(selection.x);
}

public void copy () {
	checkWidget ();
}

public Rectangle computeTrim (int x, int y, int width, int height) {
	checkWidget ();
	if ((style & SWT.MULTI) != 0) {
		ScrollControlBase widget = (ScrollControlBase)object;
		EdgeMetrics view = widget.viewMetrics;
		x -= view.left;
		y -= view.top;
		width += (view.left + view.right);
		height += (view.top + view.bottom);
	}
	return new Rectangle(x, y, width, height);
}

void createHandle () {
	if ((style & SWT.SINGLE) != 0) {
		TextInput widget = new TextInput();
		if (widget == null) error (SWT.ERROR_NO_HANDLES);
		widget.editable = (style & SWT.READ_ONLY) == 0;
		object = widget;
	} else {
		TextArea widget = new TextArea();
		if (widget == null) error (SWT.ERROR_NO_HANDLES);
		widget.editable = (style & SWT.READ_ONLY) == 0;
		object = widget;		
	}
	object.setStyle("borderStyle", hasBorder() ? "inset" : "none");
}

public void cut () {
	checkWidget ();
}

void enterHandler (FlexEvent event) {
	postEvent (SWT.DefaultSelection); 
}

native Function enterHandlerFunction ()/*{
	return enterHandler__Lmx_events_FlexEvent_2;
}*/;

public int getCaretLineNumber () {
	checkWidget ();
	return 0;
}

public Point getCaretLocation () {
	checkWidget ();
	return null;
}

public int getCaretPosition () {
	checkWidget ();
	return 0;
}

public int getCharCount () {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		return ((TextInput)object).length;
	} else {
		return ((TextArea)object).length;		
	}
}

public boolean getDoubleClickEnabled () {
	checkWidget ();
	return false;
}

public char getEchoChar () {
	checkWidget ();
	return 0;
}

public boolean getEditable () {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		return ((TextInput)object).editable;
	} else {
		return ((TextArea)object).editable;		
	}
}

public int getLineCount () {
	checkWidget ();
	return 0;
}

public String getLineDelimiter () {
	checkWidget ();
	return "\n";
}

public int getLineHeight () {
	checkWidget ();
	return 0;
}

public int getOrientation () {
	checkWidget ();
	return 0;
}

public String getMessage () {
	checkWidget ();
	return "";
}

public Point getSelection () {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		TextInput widget = (TextInput)object;
		return new Point(widget.selectionBeginIndex, widget.selectionEndIndex);
	} else {
		TextArea widget = (TextArea)object;
		return new Point(widget.selectionBeginIndex, widget.selectionEndIndex);
	}
}

public int getSelectionCount () {
	checkWidget ();
	Point selection = getSelection();
	return selection.y - selection.x;
}

public String getSelectionText () {
	checkWidget ();
	Point selection = getSelection();
	return getText(selection.x, selection.y);
}

public int getTabs () {
	checkWidget ();
	return 0;
}

public String getText () {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		return ((TextInput)object).text;
	} else {
		return ((TextArea)object).text;		
	}
}

public String getText (int start, int end) {
	checkWidget ();
	if (!(start <= end && 0 <= end)) return "";
	int length = getCharCount();
	start = Math.max (0, start);
	end = Math.min (end, length - 1);
	if ((style & SWT.SINGLE) != 0) {
		return ((TextInput)object).text.substring(start, end);
	} else {
		return ((TextArea)object).text.substring(start, end);
	}
}

public int getTextLimit () {
	checkWidget ();
	int limit;
	if ((style & SWT.SINGLE) != 0) {
		limit = ((TextInput)object).maxChars;
	} else {
		limit = ((TextArea)object).maxChars;
	}
	return limit != 0 ? limit : LIMIT;
}

public int getTopIndex () {
	checkWidget ();
	return 0;
}

public int getTopPixel () {
	checkWidget ();
	return 0;
}

void hookEvents (){
	super.hookEvents();
	object.addEventListener(FlexEvent.ENTER, enterHandlerFunction ());
}

public void insert (String string) {
	checkWidget ();
}

public void paste () {
	checkWidget ();
}

public void removeModifyListener (ModifyListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Modify, listener);	
}

public void removeSelectionListener (SelectionListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Selection, listener);
	eventTable.unhook (SWT.DefaultSelection,listener);	
}

public void removeVerifyListener (VerifyListener listener) {
	checkWidget ();
	if (listener == null) error (SWT.ERROR_NULL_ARGUMENT);
	if (eventTable == null) return;
	eventTable.unhook (SWT.Verify, listener);	
}

public void selectAll () {
	checkWidget();
	setSelection(0, getCharCount());
}

public void setDoubleClickEnabled (boolean doubleClick) {
	checkWidget();
}

public void setEchoChar (char echo) {
	checkWidget();
}

public void setEditable (boolean editable) {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		((TextInput)object).editable = editable;
	} else {
		((TextArea)object).editable = editable;		
	}
}
public void setMessage (String message) {
	checkWidget();
}

public void setOrientation (int orientation) {
	checkWidget();
}

public void setSelection (int start) {
	checkWidget ();
	setSelection(start, start);
}

public void setSelection (int start, int end) {
	checkWidget ();
	if ((style & SWT.SINGLE) != 0) {
		((TextInput)object).setSelection(start, end);
	} else {
		((TextArea)object).setSelection(start, end);
	}
}

public void setSelection (Point selection) {
	checkWidget();
}

public void setTabs (int tabs) {
	checkWidget();
}

public void setText (String string) {
	checkWidget ();
	if (string == null) error (SWT.ERROR_NULL_ARGUMENT);
	if ((style & SWT.SINGLE) != 0) {
		((TextInput)object).text = string;
	} else {
		((TextArea)object).text = string;		
	}
}

public void setTextLimit (int limit) {
	checkWidget();
	if (limit == LIMIT) limit = 0;
	if ((style & SWT.SINGLE) != 0) {
		((TextInput)object).maxChars = limit;
	} else {
		((TextArea)object).maxChars = limit;
	}
}

public void setTopIndex (int index) {
	checkWidget();
}

public void showSelection () {
	checkWidget();
}

}
